
#property copyright "Tehan Candlestick Reversal Patterns EA Copyright 2025"
#property version   "1.02"
#property description "This EA identifies and trades major candlestick reversal patterns at key Fibonacci levels"
#property description "with enhanced risk management including trailing stops and breakeven functionality."

#include <Trade\Trade.mqh>

// Pattern Selection Structure
struct PatternSettings {
    bool EnableMorningStar;
    bool EnableEveningStar;
    bool EnableDragonflyDoji;
    bool EnableGravestoneDoji;
    bool EnableShootingStar;
    bool EnableInvertedHammer;
    bool EnableBullishHarami;
    bool EnableBearishHarami;
};

// Input Parameters
input group "Trading Parameters"
input double LotSize = 0.1;
input bool UseRiskPercentage = false;
input double RiskPercentage = 1.0;
input ENUM_TIMEFRAMES FiboTimeframe = PERIOD_D1;
input bool UsePartialTP = true;

input group "Stop Management"
input double SLpips = 1000;
input bool EnableTrailingStop = true;
input double TrailingStartPips = 50.0;
input double TrailingStepPips = 10.0;
input double TrailingDistancePips = 20.0;
input bool EnableBreakeven = true;
input double BreakevenTriggerPips = 30.0;
input double BreakevenPlusPips = 5.0;

input group "Pattern Selection"
input bool Enable_MorningStar = true;
input bool Enable_EveningStar = true;
input bool Enable_DragonflyDoji = true;
input bool Enable_GravestoneDoji = true;
input bool Enable_ShootingStar = true;
input bool Enable_InvertedHammer = true;
input bool Enable_BullishHarami = true;
input bool Enable_BearishHarami = true;

input group "Fibonacci Settings"
input bool ShowSupportResistance = true;
input color SupportColor = clrGreen;
input color ResistanceColor = clrRed;
input ENUM_LINE_STYLE FiboStyle = STYLE_DOT;

input group "Pattern Settings"
input bool EnableBearishPatterns = true;
input bool TradeHangingMan = true;
input bool TradeBearishEngulfing = true;
input double SupportZoneLevel = 0.0;
input double ResistanceZoneLevel = 1.0;
input double BuyTPLevel = 0.786;
input double SellTPLevel = 0.386;

input bool EnableBullishPatterns = true;
input bool TradeHammer = true;
input bool TradeBullishEngulfing = true;

// Fibonacci Levels
double FiboLevels[] = {1.0,0.0,BuyTPLevel,SellTPLevel,-1.236,-0.5,-0.236,1.236,1.5,1.863};
double FiboValues[];
int FiboLines[];

// Global Variables
PatternSettings Patterns;
double g_point;  // Changed from 'point' to avoid naming conflict
double pip;
CTrade trade;




//+------------------------------------------------------------------+
//| Pattern Detection Functions                                      |
//+------------------------------------------------------------------+
bool IsHangingMan()
{
    double open = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close = iClose(_Symbol, PERIOD_CURRENT, 1);
    double high = iHigh(_Symbol, PERIOD_CURRENT, 1);
    double low = iLow(_Symbol, PERIOD_CURRENT, 1);
    
    double body = MathAbs(open - close);
    double lowerWick = MathMin(open, close) - low;
    double upperWick = high - MathMax(open, close);
    
    return (body < lowerWick * 0.3) && 
           (lowerWick > body * 2) && 
           (upperWick < body) && 
           IsPriceAboveResistance();
}

bool IsBearishEngulfing()
{
    double open1 = iOpen(_Symbol, PERIOD_CURRENT, 2);
    double close1 = iClose(_Symbol, PERIOD_CURRENT, 2);
    double open2 = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close2 = iClose(_Symbol, PERIOD_CURRENT, 1);
    
    return (close1 > open1) && 
           (close2 < open2) && 
           (open2 > close1) && 
           (close2 < open1) && 
           IsPriceAboveResistance();
}

bool IsHammer()
{
    double open = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close = iClose(_Symbol, PERIOD_CURRENT, 1);
    double high = iHigh(_Symbol, PERIOD_CURRENT, 1);
    double low = iLow(_Symbol, PERIOD_CURRENT, 1);
    
    double body = MathAbs(open - close);
    double lowerWick = MathMin(open, close) - low;
    double upperWick = high - MathMax(open, close);
    
    return (body < lowerWick * 0.3) && 
           (lowerWick > body * 2) && 
           (upperWick < body) && 
           IsPriceBelowSupport();
}

bool IsBullishEngulfing()
{
    double open1 = iOpen(_Symbol, PERIOD_CURRENT, 2);
    double close1 = iClose(_Symbol, PERIOD_CURRENT, 2);
    double open2 = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close2 = iClose(_Symbol, PERIOD_CURRENT, 1);
    
    return (close1 < open1) && 
           (close2 > open2) && 
           (open2 < close1) && 
           (close2 > open1) && 
           IsPriceBelowSupport();
}

bool IsDragonflyDoji()
{
    double open = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close = iClose(_Symbol, PERIOD_CURRENT, 1);
    double high = iHigh(_Symbol, PERIOD_CURRENT, 1);
    double low = iLow(_Symbol, PERIOD_CURRENT, 1);
    
    double body = MathAbs(open - close);
    double upperWick = high - MathMax(open, close);
    double lowerWick = MathMin(open, close) - low;
    
    bool isDoji = body < (high - low) * 0.1;
    bool hasLongLowerWick = lowerWick > body * 3;
    bool hasShortUpperWick = upperWick < body;
    
    return isDoji && hasLongLowerWick && hasShortUpperWick && IsPriceBelowSupport();
}

bool IsInvertedHammer()
{
    double open = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close = iClose(_Symbol, PERIOD_CURRENT, 1);
    double high = iHigh(_Symbol, PERIOD_CURRENT, 1);
    double low = iLow(_Symbol, PERIOD_CURRENT, 1);
    
    double body = MathAbs(open - close);
    double upperWick = high - MathMax(open, close);
    double lowerWick = MathMin(open, close) - low;
    
    return (upperWick > body * 2) && 
           (lowerWick < body * 0.1) && 
           (body < (high - low) * 0.3) &&
           IsPriceBelowSupport();
}

bool IsBullishHarami()
{
    double open1 = iOpen(_Symbol, PERIOD_CURRENT, 2);
    double close1 = iClose(_Symbol, PERIOD_CURRENT, 2);
    double open2 = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close2 = iClose(_Symbol, PERIOD_CURRENT, 1);
    
    bool firstBearish = close1 < open1;
    bool secondBullish = close2 > open2;
    bool insidePrevious = open2 > close1 && close2 < open1;
    
    return firstBearish && secondBullish && insidePrevious && IsPriceBelowSupport();
}

bool IsGravestoneDoji()
{
    double open = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close = iClose(_Symbol, PERIOD_CURRENT, 1);
    double high = iHigh(_Symbol, PERIOD_CURRENT, 1);
    double low = iLow(_Symbol, PERIOD_CURRENT, 1);
    
    double body = MathAbs(open - close);
    double upperWick = high - MathMax(open, close);
    double lowerWick = MathMin(open, close) - low;
    
    bool isDoji = body < (high - low) * 0.1;
    bool hasLongUpperWick = upperWick > body * 3;
    bool hasShortLowerWick = lowerWick < body;
    
    return isDoji && hasLongUpperWick && hasShortLowerWick && IsPriceAboveResistance();
}

bool IsShootingStar()
{
    double open = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close = iClose(_Symbol, PERIOD_CURRENT, 1);
    double high = iHigh(_Symbol, PERIOD_CURRENT, 1);
    double low = iLow(_Symbol, PERIOD_CURRENT, 1);
    
    double body = MathAbs(open - close);
    double upperWick = high - MathMax(open, close);
    double lowerWick = MathMin(open, close) - low;
    
    return (upperWick > body * 2) && 
           (lowerWick < body * 0.1) && 
           (body < (high - low) * 0.3) &&
           IsPriceAboveResistance();
}

bool IsBearishHarami()
{
    double open1 = iOpen(_Symbol, PERIOD_CURRENT, 2);
    double close1 = iClose(_Symbol, PERIOD_CURRENT, 2);
    double open2 = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close2 = iClose(_Symbol, PERIOD_CURRENT, 1);
    
    bool firstBullish = close1 > open1;
    bool secondBearish = close2 < open2;
    bool insidePrevious = open2 < close1 && close2 > open1;
    
    return firstBullish && secondBearish && insidePrevious && IsPriceAboveResistance();
}
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
    SetChartAppearance();
    
    // Initialize pattern settings
    Patterns.EnableMorningStar = Enable_MorningStar;
    Patterns.EnableEveningStar = Enable_EveningStar;
    Patterns.EnableDragonflyDoji = Enable_DragonflyDoji;
    Patterns.EnableGravestoneDoji = Enable_GravestoneDoji;
    Patterns.EnableShootingStar = Enable_ShootingStar;
    Patterns.EnableInvertedHammer = Enable_InvertedHammer;
    Patterns.EnableBullishHarami = Enable_BullishHarami;
    Patterns.EnableBearishHarami = Enable_BearishHarami;
    
    // Initialize Fibonacci arrays
    ArrayResize(FiboValues, ArraySize(FiboLevels));
    ArrayResize(FiboLines, ArraySize(FiboLevels));
    
    // Initialize point and pip values
    g_point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
    pip = g_point * 10;
    
    return(INIT_SUCCEEDED);
}
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
    DeleteFiboLines();
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
    if(IsNewBar())
    {
        CalculateFiboLevels();
        if(ShowSupportResistance)
            DrawFiboLevels();
            
        CheckPatterns();
    }
    
    ManageRisk();
}

//+------------------------------------------------------------------+
//| Pattern Detection Function                                       |
//+------------------------------------------------------------------+
void CheckPatterns()
{
    // Check bearish patterns
    if(EnableBearishPatterns)
    {
        if(TradeHangingMan && IsHangingMan())
            ExecuteSellOrder("Hanging Man");
            
        if(TradeBearishEngulfing && IsBearishEngulfing())
            ExecuteSellOrder("Bearish Engulfing");
    }
    
    // Check bullish patterns
    if(EnableBullishPatterns)
    {
        if(TradeHammer && IsHammer())
            ExecuteBuyOrder("Hammer");
            
        if(TradeBullishEngulfing && IsBullishEngulfing())
            ExecuteBuyOrder("Bullish Engulfing");
    }
        
    // Check other patterns
    if(Patterns.EnableMorningStar && IsMorningStar())
        ExecuteBuyOrder("Morning Star");
    else if(Patterns.EnableDragonflyDoji && IsDragonflyDoji())
        ExecuteBuyOrder("Dragonfly Doji");
    else if(Patterns.EnableInvertedHammer && IsInvertedHammer())
        ExecuteBuyOrder("Inverted Hammer");
    else if(Patterns.EnableBullishHarami && IsBullishHarami())
        ExecuteBuyOrder("Bullish Harami");
    else if(Patterns.EnableEveningStar && IsEveningStar())
        ExecuteSellOrder("Evening Star");
    else if(Patterns.EnableGravestoneDoji && IsGravestoneDoji())
        ExecuteSellOrder("Gravestone Doji");
    else if(Patterns.EnableShootingStar && IsShootingStar())
        ExecuteSellOrder("Shooting Star");
    else if(Patterns.EnableBearishHarami && IsBearishHarami())
        ExecuteSellOrder("Bearish Harami");
}

//+------------------------------------------------------------------+
//| Risk Management Function                                         |
//+------------------------------------------------------------------+
void ManageRisk()
{
    if(EnableTrailingStop)
        ManageTrailingStops();
        
    if(EnableBreakeven)
        ManageBreakeven();
        
    if(UsePartialTP)
        ManagePartialTakeProfit();
}

//+------------------------------------------------------------------+
//| Pattern Detection Functions                                      |
//+------------------------------------------------------------------+
bool IsMorningStar()
{
    double open1 = iOpen(_Symbol, PERIOD_CURRENT, 3);
    double close1 = iClose(_Symbol, PERIOD_CURRENT, 3);
    double open2 = iOpen(_Symbol, PERIOD_CURRENT, 2);
    double close2 = iClose(_Symbol, PERIOD_CURRENT, 2);
    double open3 = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close3 = iClose(_Symbol, PERIOD_CURRENT, 1);
    
    bool firstBearish = close1 < open1;
    bool smallMiddle = MathAbs(open2 - close2) < MathAbs(open1 - close1) * 0.3;
    bool lastBullish = close3 > open3;
    
    return firstBearish && smallMiddle && lastBullish && IsPriceBelowSupport();
}

bool IsEveningStar()
{
    double open1 = iOpen(_Symbol, PERIOD_CURRENT, 3);
    double close1 = iClose(_Symbol, PERIOD_CURRENT, 3);
    double open2 = iOpen(_Symbol, PERIOD_CURRENT, 2);
    double close2 = iClose(_Symbol, PERIOD_CURRENT, 2);
    double open3 = iOpen(_Symbol, PERIOD_CURRENT, 1);
    double close3 = iClose(_Symbol, PERIOD_CURRENT, 1);
    
    bool firstBullish = close1 > open1;
    bool smallMiddle = MathAbs(open2 - close2) < MathAbs(open1 - close1) * 0.3;
    bool lastBearish = close3 < open3;
    
    return firstBullish && smallMiddle && lastBearish && IsPriceAboveResistance();
}


//+------------------------------------------------------------------+
//| Trade Execution Functions                                        |
//+------------------------------------------------------------------+
void ExecuteBuyOrder(string pattern)
{
    double stopLoss = iLow(_Symbol, PERIOD_CURRENT, 1) - SLpips * _Point;
    double takeProfit = FindBuyTPZone();
    double lots = CalculateLotSize(SymbolInfoDouble(_Symbol, SYMBOL_BID) - stopLoss);
    
    trade.Buy(lots, _Symbol, 0, stopLoss, takeProfit, pattern);
}

void ExecuteSellOrder(string pattern)
{
    double stopLoss = iHigh(_Symbol, PERIOD_CURRENT, 1) + SLpips * _Point;
    double takeProfit = FindSellTPZone();
    double lots = CalculateLotSize(stopLoss - SymbolInfoDouble(_Symbol, SYMBOL_ASK));
    
    trade.Sell(lots, _Symbol, 0, stopLoss, takeProfit, pattern);
}

//+------------------------------------------------------------------+
//| Risk Management Functions                                        |
//+------------------------------------------------------------------+
void ManageTrailingStops()
{
    for(int i = PositionsTotal() - 1; i >= 0; i--)
    {
        if(PositionSelectByTicket(PositionGetTicket(i)))
        {
            if(_Symbol != PositionGetString(POSITION_SYMBOL))
                continue;
                
            double positionOpenPrice = PositionGetDouble(POSITION_PRICE_OPEN);
            double currentStopLoss = PositionGetDouble(POSITION_SL);
            double currentPrice = PositionGetDouble(POSITION_PRICE_CURRENT);
            double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
            double trailingStart = TrailingStartPips * 10 * point;
            double trailingStep = TrailingStepPips * 10 * point;
            double trailingDistance = TrailingDistancePips * 10 * point;
            
            if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL)
            {
                double newStopLoss = currentPrice + trailingDistance;
                if(positionOpenPrice - currentPrice >= trailingStart)
                {
                    if(currentStopLoss == 0 || 
                       (newStopLoss < currentStopLoss && currentPrice + trailingDistance < currentStopLoss - trailingStep))
                    {
                        trade.PositionModify(PositionGetTicket(i), newStopLoss, PositionGetDouble(POSITION_TP));
                    }
                }
            }
            else if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
            {
                double newStopLoss = currentPrice - trailingDistance;
                if(currentPrice - positionOpenPrice >= trailingStart)
                {
                    if(currentStopLoss == 0 || 
                       (newStopLoss > currentStopLoss && currentPrice - trailingDistance > currentStopLoss + trailingStep))
                    {
                        trade.PositionModify(PositionGetTicket(i), newStopLoss, PositionGetDouble(POSITION_TP));
                    }
                }
            }
        }
    }
}

void ManageBreakeven()
{
    for(int i = PositionsTotal() - 1; i >= 0; i--)
    {
        if(PositionSelectByTicket(PositionGetTicket(i)))
        {
            if(_Symbol != PositionGetString(POSITION_SYMBOL))
                continue;
                
            double positionOpenPrice = PositionGetDouble(POSITION_PRICE_OPEN);
            double currentStopLoss = PositionGetDouble(POSITION_SL);
            double currentPrice = PositionGetDouble(POSITION_PRICE_CURRENT);
            double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
            double breakevenTrigger = BreakevenTriggerPips * 10 * point;
            double breakevenPlus = BreakevenPlusPips * 10 * point;
            
            if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL)
            {
                if(positionOpenPrice - currentPrice >= breakevenTrigger)
                {
                    if(currentStopLoss == 0 || currentStopLoss > positionOpenPrice + breakevenPlus)
                    {
                        double newStopLoss = positionOpenPrice + breakevenPlus;
                        trade.PositionModify(PositionGetTicket(i), newStopLoss, PositionGetDouble(POSITION_TP));
                    }
                }
            }
            else if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
            {
                if(currentPrice - positionOpenPrice >= breakevenTrigger)
                {
                    if(currentStopLoss == 0 || currentStopLoss < positionOpenPrice - breakevenPlus)
                    {
                        double newStopLoss = positionOpenPrice - breakevenPlus;
                        trade.PositionModify(PositionGetTicket(i), newStopLoss, PositionGetDouble(POSITION_TP));
                    }
                }
            }
        }
    }
}

void ManagePartialTakeProfit()
{
    double dailyhigh = iHigh(_Symbol,FiboTimeframe,1);
    double dailylow = iLow(_Symbol,FiboTimeframe,1);
    double range = dailyhigh - dailylow;
    double Fibo50Level = NormalizeDouble(dailyhigh - range * 0.5,_Digits);
    
    for(int i = PositionsTotal() - 1; i >= 0; i--)
    {
        if(PositionSelectByTicket(PositionGetTicket(i)))
        {
            if(_Symbol != PositionGetString(POSITION_SYMBOL))
                continue;
                
            double positionVolume = PositionGetDouble(POSITION_VOLUME);
            double currentPrice = PositionGetDouble(POSITION_PRICE_CURRENT);
            
            if(positionVolume == LotSize)
            {
                if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL)
                {
                    if(currentPrice < Fibo50Level)
                    {
                        trade.PositionClosePartial(PositionGetTicket(i), positionVolume/2, -1);
                    }
                }
                else if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
                {
                    if(currentPrice > Fibo50Level)
                    {
                        trade.PositionClosePartial(PositionGetTicket(i), positionVolume/2, -1);
                    }
                }
            }
        }
    }
}

//+------------------------------------------------------------------+
//| Helper Functions                                                 |
//+------------------------------------------------------------------+
double CalculateLotSize(double stopDistance)
{
    if(UseRiskPercentage)
    {
        double accountBalance = AccountInfoDouble(ACCOUNT_BALANCE);
        double riskAmount = accountBalance * RiskPercentage / 100;
        double tickValue = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_TICK_VALUE);
        double tickSize = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_TICK_SIZE);
        
        return NormalizeDouble(riskAmount / (stopDistance * tickValue / tickSize), 2);
    }
    return LotSize;
}

bool IsNewBar()
{
    static datetime lastBar;
    datetime currentBar = iTime(_Symbol, PERIOD_CURRENT, 0);
    
    if(lastBar != currentBar)
    {
        lastBar = currentBar;
        return true;
    }
    return false;
}

bool IsPriceAboveResistance()
{
    return SymbolInfoDouble(_Symbol, SYMBOL_BID) > FiboValues[ArraySize(FiboValues)-2];
}

bool IsPriceBelowSupport()
{
    return SymbolInfoDouble(_Symbol, SYMBOL_ASK) < FiboValues[1];
}

void DeleteFiboLines()
{
    for(int i = 0; i < ArraySize(FiboLines); i++)
    {
        ObjectDelete(0, "FiboLine" + IntegerToString(i));
    }
}

void CalculateFiboLevels()
{
    double highPrice = iHigh(_Symbol, FiboTimeframe, 1);
    double lowPrice = iLow(_Symbol, FiboTimeframe, 1);
    double range = highPrice - lowPrice;
    
    for(int i = 0; i < ArraySize(FiboLevels); i++)
    {
        FiboValues[i] = lowPrice + (range * FiboLevels[i]);
    }
}

void DrawFiboLevels()
{
    DeleteFiboLines();
    
    for(int i = 0; i < ArraySize(FiboLevels); i++)
    {
        string lineName = "FiboLine" + IntegerToString(i);
        FiboLines[i] = ObjectCreate(0, lineName, OBJ_HLINE, 0, 0, FiboValues[i]);
        
        if(FiboLines[i])
        {
            ObjectSetInteger(0, lineName, OBJPROP_STYLE, FiboStyle);
            ObjectSetInteger(0, lineName, OBJPROP_COLOR, GetFiboColor(FiboLevels[i]));
            ObjectSetString(0, lineName, OBJPROP_TEXT, GetFiboLabel(FiboLevels[i]));
        }
    }
}

color GetFiboColor(double level)
{
    if(level == SupportZoneLevel)
        return SupportColor;
    if(level == ResistanceZoneLevel)
        return ResistanceColor;
    return clrGray;
}

string GetFiboLabel(double level)
{
    if(level == SupportZoneLevel)
        return "Support Zone";
    if(level == ResistanceZoneLevel)
        return "Resistance Zone";
    return DoubleToString(level, 3);
}

double FindSellTPZone()
{
    for(int i = 0; i < ArraySize(FiboLevels); i++)
    {
        if(FiboLevels[i] == SellTPLevel)
            return FiboValues[i];
    }
    return 0;
}

double FindBuyTPZone()
{
    for(int i = 0; i < ArraySize(FiboLevels); i++)
    {
        if(FiboLevels[i] == BuyTPLevel)
            return FiboValues[i];
    }
    return 0;
}

void SetChartAppearance()
{
    ChartSetInteger(0, CHART_COLOR_BACKGROUND, clrBlack);
    ChartSetInteger(0, CHART_COLOR_FOREGROUND, clrWhite);
    ChartSetInteger(0, CHART_COLOR_GRID, clrBlack);
    ChartSetInteger(0, CHART_COLOR_CHART_UP, clrGreen);
    ChartSetInteger(0, CHART_COLOR_CHART_DOWN, clrRed);
    ChartSetInteger(0, CHART_COLOR_CANDLE_BULL, clrGreen);
    ChartSetInteger(0, CHART_COLOR_CANDLE_BEAR, clrRed);
    ChartSetInteger(0, CHART_COLOR_VOLUME, clrGray);
    ChartSetInteger(0, CHART_COLOR_CHART_LINE, clrWhite);
    ChartSetInteger(0, CHART_SHOW_GRID, false);
    ChartSetInteger(0, CHART_SHOW_PERIOD_SEP, true);
    ChartSetInteger(0,CHART_SHOW_OBJECT_DESCR,true);
    ChartSetInteger(0,CHART_SHOW_VOLUMES,false);
}
//+------------------------------------------------------------------+